<?php

use League\Csv\Reader as CsvReader;
use League\Csv\Writer as CsvWriter;
use Laminas\Json\Decoder;
use Laminas\Json\Encoder;
use Laminas\Json\Json;
use Symfony\Component\VarDumper\VarDumper;
use Ramsey\Uuid\Uuid;

require_once 'bootstrap.php';

set_error_handler(function ($severity, $message, $file, $line) {
    echo $message."\n";
    throw new \ErrorException($message, $severity, $severity, $file, $line);
});

define('STATUS_FILE', __DIR__ . '/running.csv');
define('RESULT_FILE', __DIR__ . '/excecution-result.csv');

$detailReader = CsvReader::createFromPath(__DIR__ . '/excecution-detail.csv');
$detailReader->setHeaderOffset(0);
$detailReader->setDelimiter(',');

$count = $detailReader->count();

if (isRunning()) {
    echo "Another process is running!";
    exit();
}

//setAsRunning();

for ($i = 0; $i < $count; $i++) {
    $executionDetail = $detailReader->fetchOne($i);
    $executionDetail['saveDestination'] = __DIR__ . '/cron-output';
    
    if(isJobCompleted($executionDetail)){
        echo 'Completed Job for' . $executionDetail["sn"]."\n";
        continue;
    }
    
    $allCompleted = false;
    for ($iterationNumber = 0; $iterationNumber < $executionDetail['repetitions']; $iterationNumber++) {
        try {
            echo "Running : \n";
            VarDumper::dump($executionDetail);
            $result = process($executionDetail, $iterationNumber);
            echo "Completed ! \n\n";
        } catch (\Exception $ex) {
            echo "Failed at iteration $iterationNumber \n"
                    . "Re executing ... \n";
            $iterationNumber--;
        }
    }
    setJobCompleted($executionDetail);
}

function isJobCompleted($executionDetail){
    $resultReader = CsvReader::createFromPath(RESULT_FILE);
    $resultReader->setHeaderOffset(0);
    $count = $resultReader->count();
    for($i=0;$i<$count;$i++){
        $result = $resultReader->fetchOne($i);
        if($executionDetail['sn'] == $result['sn'] && $result['completed'] ==1 ){
            return true;
        }
    }
    return false;
}

function setJobCompleted($executionDetail){
    $executionDetail['completed'] = 1;
    $resultReader = CsvReader::createFromPath(RESULT_FILE);
    $resultWriter = CsvWriter::createFromPath(RESULT_FILE);
    $oldDetails = $resultReader->getRecords();
    foreach($oldDetails as $o){
        $resultWriter->insertOne($o);
    }
    $resultWriter->insertOne($executionDetail);
    return ;
}

function getSaveDetination($executionDetail, $iterationNumber){
    return $executionDetail['saveDestination'] . '/'
            . 'SN_' . $executionDetail['sn'] . '__'
            . $executionDetail['objective'] . '_'
            . '__iterationCount_' . $executionDetail['numberOfIterations']
            . '__runNo_' . $iterationNumber . '';
}

function process($executionDetail, $iterationNumber): array {
    $objectiveEvaluator = $executionDetail['objective'];
    $initializationType = $executionDetail['initializationType'];
    $decisionSpaceDimension = $executionDetail['dimension'];
    $partsPerDimension = $executionDetail['partsPerDimension'];
    $hyperCubeDimension = $executionDetail['hyperCubeDimension'] == '__null__' ? $decisionSpaceDimension : $executionDetail['hyperCubeDimension'];
    $lowerBound = $executionDetail['lowerBound'];
    $upperBound = $executionDetail['upperBound'];
    $symbolObjectPerMembrane = $executionDetail['symbolObjectPerMembrane'];
    $eliteMatePercentage = $executionDetail['eliteMatePercentage'];
    $mutationChance = $executionDetail['mutationChance'];
    $preservePercent = 50;

    $thresholds = explode('/', $executionDetail['mergeDivideThreshold']);
    $mergeThreshold = $thresholds[0];
    $divideThreshold = $thresholds[1];

    $numberOfIterations = $executionDetail['numberOfIterations'];

    $saveDestination = getSaveDetination($executionDetail, $iterationNumber);
    if(is_dir($saveDestination)){
        rename($saveDestination, $saveDestination.'_failed_'.Uuid::uuid4());
    }
    mkdir($saveDestination);

    $startTime = microtime(true);

    optimize(
            $objectiveEvaluator,
            $decisionSpaceDimension,
            $lowerBound,
            $upperBound,
            $hyperCubeDimension,
            $partsPerDimension,
            $symbolObjectPerMembrane,
            $initializationType,
            $numberOfIterations,
            $eliteMatePercentage,
            $preservePercent,
            $mergeThreshold,
            $divideThreshold,
            $mutationChance,
            true,
            $saveDestination
    );
    $endTime = microtime(true);
    //cleanFronts($saveDestination);
    return [];
}

function setAsRunning() {
    unlink(STATUS_FILE);
    $statusWriter = CsvWriter::createFromPath(STATUS_FILE, 'w');
    $statusWriter->insertAll([['isRunning'], [1]]);
}

function setAsStopped() {
    unlink(STATUS_FILE);
    $statusWriter = CsvWriter::createFromPath(STATUS_FILE, 'w');
    $statusWriter->insertAll([['isRunning'], [0]]);
}

function isRunning() {
    if (!is_file(STATUS_FILE)) {
        setAsStopped();
    }
    $statusReader = CsvReader::createFromPath(STATUS_FILE);
    $statusReader->setHeaderOffset(0);
    $status = $statusReader->fetchOne();
    return $status['isRunning'] == 1;
}

function cleanFronts($saveDestination) {
    $executionDetail = Decoder::decode(file_get_contents($saveDestination . '/execution-detail.json'), Json::TYPE_ARRAY);
    $iteration = $executionDetail['numberOfIterations'];

    for ($iterationNumber = 0; $iterationNumber < $iteration; $iterationNumber++) {
        $symbolObjFile = $saveDestination . '/iteration-'.($iterationNumber+1).'-symbolobjects.json';
        $frontFile = $saveDestination . '/iteration-'.($iterationNumber+1).'-fronts.json';
        $membraneFile = $saveDestination . '/iteration-'.($iterationNumber+1).'-membranes.json';
        
        $symbolObjects = Decoder::decode(file_get_contents($symbolObjFile), Json::TYPE_ARRAY);
        $rejects = getRejectedIds($symbolObjects, getLimit($iterationNumber));
        
        foreach($symbolObjects as $key => $val){
            if(in_array($key, $rejects)){
                unset($symbolObjects[$key]);
            }
        }
        file_put_contents($symbolObjFile, Encoder::encode($symbolObjects));
        
        $fronts = Decoder::decode(file_get_contents($frontFile), Json::TYPE_ARRAY);
        foreach($fronts as $frontNo => $ids){
            foreach($ids as $key => $id){
                if(in_array($id, $rejects)){
                    unset($fronts[$frontNo][$key]);
                }
            }
        }
        file_put_contents($frontFile, Encoder::encode($fronts));
        
        $membranes = Decoder::decode(file_get_contents($membraneFile), Json::TYPE_ARRAY);
        foreach($membranes as $mId => $m){
            foreach($m['symbolObjects'] as $id => $s){
                if(in_array($id, $rejects)){
                    unset($membranes[$mId]['symbolObjects'][$id]);
                }
            }
        }
        file_put_contents($membraneFile, Encoder::encode($membranes));
    }
}

function getRejectedIds($symbolObjects, $limit){
    $reject = [];
    foreach ($symbolObjects as $s){
        $objectives = $s['objectives'];
        foreach($objectives as $o){
            if(abs($o) > $limit){
                $reject[] = $s['id'];
                break;
            }
        }
    }
    return $reject;
}

function getLimit($iterationNumber){
    $setting = [
        ["iteration" => 5, "max" => 10],
        ["iteration" => 10, "max" => 7.5],
        ["iteration" => 15, "max" => 5],
        ["iteration" => 20, "max" => 2.5],
        ["iteration" => 25, "max" => 1.1],
    ];
    foreach($setting as $s){
        if($iterationNumber <= $s["iteration"] ){
            return $s["max"];
        }
    }
    return $s["max"];
}