<?php

CONST MAX_HYPER_VOLUME_DIMENSION = 5;
CONST MAX_NUMBER_OF_MEMBRANES = 1000;

/**
 * 
 * Compartmentalizes the decision space into hyper-volumes(membranes). 
 * Returns hyper-volume array with lower and upper bound 
 * 
 * @param int $decisionSpaceDimension
 * @param int $partsPerDimension
 * @param int $hyperCubeDimension
 * @param array|float $lowerBound
 * @param array|float $upperBound
 * 
 * @return array
 */
function createMembranes(
        int $decisionSpaceDimension,
        int $partsPerDimension,
        int $hyperCubeDimension,
        array|float $lowerBound,
        array|float $upperBound
): array {

    $hyperCubeDimension = min($hyperCubeDimension, MAX_HYPER_VOLUME_DIMENSION);
    $membranes = [];
    $increments = []; //Dimension wise increment
    $noOfMembranes = pow($partsPerDimension, $hyperCubeDimension);

    //If dimension wise range is not provided, give same range to all the dimensions
    if (!is_array($lowerBound)) {
        $lowerBound = array_fill(0, $decisionSpaceDimension, $lowerBound);
    }
    //If dimension wise range is not provided, give same range to all the dimensions
    if (!is_array($upperBound)) {
        $upperBound = array_fill(0, $decisionSpaceDimension, $upperBound);
    }

    //Increment for each dimension
    for ($i = 0; $i < $hyperCubeDimension; $i++) {
        $increments[$i] = ($upperBound[$i] - $lowerBound[$i]) / $partsPerDimension;
    }
    
    for ($n = 0; $n < $noOfMembranes; $n++) {
        $membranes[$n] = [
            'id' => $n,
            'linearPosition' => $n,
            'hyperCubePosition' => convertLinearPositionToHyperCubePosition($n, $partsPerDimension, $hyperCubeDimension),
            'symbolObjects' => [],
            'range' => [
                'lower' => $lowerBound,
                'upper' => $upperBound,
            ],
            'mergedMembraneIds' => [],
        ];
        $membranes[$n]['neighbours'] = getNeighbouringMembranePosition(
                $membranes[$n]['hyperCubePosition'],
                $partsPerDimension,
                true
        );
        
        //Set the range is membrane 
        for ($o = 0; $o < $hyperCubeDimension; $o++) {
            $membranes[$n]['range']['lower'][$o] = $membranes[$n]['hyperCubePosition'][$o]*$increments[$o];
            $membranes[$n]['range']['upper'][$o] = $membranes[$n]['hyperCubePosition'][$o]*$increments[$o]+$increments[$o];
        }
    }
    return $membranes;
}

function convertHyperCubePositionToLinearPosition(
        array $hyperCubePosition,
        int $partitionCount
): int {
    $linearPosition = 0;
    $multiplier = 1;
    for ($i = 0; $i < $partitionCount; $i++) {
        if (!isset($hyperCubePosition[$i])) {
            break;
        }
        $linearPosition += $multiplier * $hyperCubePosition[$i];
        $multiplier *= $partitionCount;
    }
    return $linearPosition;
}

function convertLinearPositionToHyperCubePosition(int $linearPosition, int $partitionCount, int $hyperCubeDimension): array {
    $hyperCubePosition = array_fill(0, $hyperCubeDimension, 0);
    $dimensionCount = 0;

    do {
        $remainder = $linearPosition % $partitionCount;
        $linearPosition = (int) ($linearPosition - $remainder) / $partitionCount;
        $hyperCubePosition[$dimensionCount] = $remainder;
        $dimensionCount++;
    } while ($linearPosition > 0);
    return $hyperCubePosition;
}

function getNeighbouringMembranePosition(
        array $currentHyperCuberPosition,
        int $partitionCount,
        bool $getLinearPosition = false
): array {
    $neighbours = [];
    $processedDimension = [];
    foreach ($currentHyperCuberPosition as $dimPosition => $p) {
        if (in_array($dimPosition, $processedDimension)) {
            continue;
        }
        if ($p > 0) {
            $cloned = $currentHyperCuberPosition;
            $cloned[$dimPosition] -= 1;
            $neighbours[] = $getLinearPosition ? convertHyperCubePositionToLinearPosition($cloned, $partitionCount) : $cloned;
        }
        if ($p < $partitionCount - 1) {
            $cloned = $currentHyperCuberPosition;
            $cloned[$dimPosition] += 1;
            $neighbours[] = $getLinearPosition ? convertHyperCubePositionToLinearPosition($cloned, $partitionCount) : $cloned;
        }
        $processedDimension[] = $dimPosition;
    }
    return $neighbours;
}
