<?php

/**
 * Repopulate membranes with symbol objects according to global pareto front.
 * 
 * @param array $membranes
 * @param array $symbolObjects
 * @param array $fronts
 * @param float $preservePercent
 * @return void
 */
function repopulate(array &$membranes, array $symbolObjects, array $fronts, float $preservePercent = 50): void {
    $total = sizeof($symbolObjects);
    $preserveCount = (int) ceil($total * $preservePercent) / 100;
    $fullyPreservedFronts = [];
    $symbolObjectCount = [];
    $partiallyPreservedFront = null;
    $partiallyPreservedCount = null;

    //Find out which fronts will be fully prreserved and which fronts will be partially preserved
    $cumulativeCount = 0;
    foreach ($fronts as $frontNo => $f) {
        $cumulativeCount += sizeof($f);
        if ($cumulativeCount < $preserveCount) {
            $fullyPreservedFronts[] = $frontNo;
        } else {
            if ($cumulativeCount > $preserveCount) {
                $partiallyPreservedFront = $frontNo;
                $partiallyPreservedCount = $preserveCount - ($cumulativeCount - sizeof($f));
            }
            break;
        }
    }

    //send preserved fronts to membranes
    foreach ($fullyPreservedFronts as $f) {
        foreach ($fronts[$f] as $symbolObjectId) {
            $obj = $symbolObjects[$symbolObjectId];
            $membranes[$obj['membraneId']]['symbolObjects'][] = $obj;

            if (!isset($symbolObjectCount[$obj['membraneId']])) {
                $symbolObjectCount[$obj['membraneId']] = 0;
            }
            $symbolObjectCount[$obj['membraneId']]++;
        }
    }

    //Return if there is no partially preserved front
    if (is_null($partiallyPreservedFront)) {
        return;
    }

    //determine which symbol objects to preserve from partially preserved front
    $sourceMembraneIds = [];
    $preservedCountPerMembrane = [];
    $partiallyPreservedCounter = 0;
    $max = 1;
    foreach ($fronts[$partiallyPreservedFront] as $symbolObjectId) {
        $membraneId = $symbolObjects[$symbolObjectId]['membraneId'];
        $sourceMembraneIds[$membraneId][] = $symbolObjectId;
        $max = max($max, sizeof($sourceMembraneIds[$membraneId]));
    }

    
    for ($j = 0; $j < $max; $j++) {
        foreach ($sourceMembraneIds as $membraneId => $symbolObjectIds) {
            if (!isset($symbolObjectIds[$j])) {
                continue;
            }
            $membranes[$membraneId]['symbolObjects'][] = $symbolObjects[$symbolObjectIds[$j]];
            $partiallyPreservedCounter++;
            if($partiallyPreservedCounter >= $partiallyPreservedCount){
                break 2;
            }
        }
    }

    return;
}
