<?php

use Symfony\Component\VarDumper\VarDumper;

function optimize(
        string $objectiveEvaluator,
        int $decisionSpaceDimension,
        array|float $lowerBound,
        array|float $upperBound,
        int $hyperCubeDimension = null,
        int $partsPerDimension = 4,
        int $symbolObjectPerMembrane = 40,
        string $initializationType = null,
        int $numberOfIterations = 20,
        float $eliteMatePercentage = 80,
        float $preservePercent = 50,
        float $thresholdForMerge = 50,
        float $thresholdForDivision = 150,
        float $mutationChance = 0.01,
        bool $saveProgress = false,
        string $progressFileLocation = null
): array {
    if (is_null($hyperCubeDimension)) {
        $hyperCubeDimension = min(MAX_HYPER_VOLUME_DIMENSION, $decisionSpaceDimension);
    }

    if (is_null($initializationType)) {
        $initializationType = LATINHYPERCUBEINITIALIZATION;
    }

    $initialMembraneCount = pow($partsPerDimension, $hyperCubeDimension);

    $membranes = createMembranes(
            $decisionSpaceDimension,
            $partsPerDimension,
            $hyperCubeDimension,
            $lowerBound,
            $upperBound
    );

    initializeMembranes(
            $membranes,
            $initializationType,
            $objectiveEvaluator,
            $decisionSpaceDimension,
            $symbolObjectPerMembrane
    );

    if ($saveProgress && is_dir($progressFileLocation)) {
        $executionDetail = [
            "objectiveEvaluator" => $objectiveEvaluator,
            "decisionSpaceDimension" => $decisionSpaceDimension,
            "lowerBound" => $lowerBound,
            "upperBound" => $upperBound,
            "hyperCubeDimension" => $hyperCubeDimension,
            "partsPerDimension" => $partsPerDimension,
            "symbolObjectPerMembrane" => $symbolObjectPerMembrane,
            "initializationType" => $initializationType,
            "numberOfIterations" => $numberOfIterations,
            "eliteMatePercentage" => $eliteMatePercentage,
            "preservePercent" => $preservePercent,
            "thresholdForMerge" => $thresholdForMerge,
            "thresholdForDivision" => $thresholdForDivision,
            "mutationChance" => $mutationChance,
            "saveProgress" => $saveProgress,
            "progressFileLocation" => $progressFileLocation
        ];
        file_put_contents(
                $progressFileLocation.'/execution-detail.json', 
                \Laminas\Json\Encoder::encode($executionDetail)
        );
        
        file_put_contents(
                $progressFileLocation.'/initial-membranes.json', 
                \Laminas\Json\Encoder::encode($membranes)
        );
    }

    $iterationCounter = 0;
    do {
        //echo "Iteration : $iterationCounter<br/>";
        foreach ($membranes as $membraneId => $mem) {
            $pairings = assignMate($mem, $eliteMatePercentage);
            $membranes[$membraneId]['pairings'] = $pairings;
        }

        foreach ($membranes as $membraneId => $mem) {
            $children = createOffsprings($mem, $objectiveEvaluator, $mutationChance);
            $membranes[$membraneId]['children'] = $children;
        }

        $extracted = extractSymbolObjects($membranes);
        
        $fronts = sortPopulationByNonDomination($extracted);
        
        repopulate($membranes, $extracted, $fronts, $preservePercent);
        
        divide($membranes, $symbolObjectPerMembrane, $thresholdForDivision);
        
        merge($membranes, $symbolObjectPerMembrane, $initialMembraneCount, $thresholdForMerge);
        
        $iterationCounter++;
        foreach ($membranes as $id => $mem) {
            $membranes[$id]['fronts'] = sortPopulationByNonDomination($mem['symbolObjects']);
        }
        
        if ($saveProgress && is_dir($progressFileLocation)) {
            file_put_contents(
                    $progressFileLocation.'/iteration-'.$iterationCounter.'-membranes.json', 
                    \Laminas\Json\Encoder::encode($membranes)
            );
            file_put_contents(
                    $progressFileLocation.'/iteration-'.$iterationCounter.'-symbolobjects.json', 
                    \Laminas\Json\Encoder::encode($extracted)
            );
            file_put_contents(
                    $progressFileLocation.'/iteration-'.$iterationCounter.'-fronts.json', 
                    \Laminas\Json\Encoder::encode($fronts)
            );
        }
        
    } while ($iterationCounter < $numberOfIterations);

    //displayFronts($fronts, $extracted);

    return [];
}

function initializeMembranes(&$membranes, $initializationType, $objectiveEvaluator, $decisionSpaceDimension, $symbolObjectPerMembrane) {
    $membraneIds = array_keys($membranes);
    $symbolObjCount = 0;
    foreach ($membraneIds as $id) {
        $range = $membranes[$id]['range'];
        $symbolObjs = [];
        $initializationValues = getInitializationValues(
                $initializationType,
                $symbolObjectPerMembrane,
                $decisionSpaceDimension,
                $range['lower'],
                $range['upper']
        );

        foreach ($initializationValues as $in) {
            $symbolObjs[$symbolObjCount] = [
                'id' => $symbolObjCount,
                'membraneId' => $id,
                'coordinate' => $in,
                'objectives' => getObjectiveValues($objectiveEvaluator, $in),
            ];
            $symbolObjCount++;
        }
        $membranes[$id]['symbolObjects'] = $symbolObjs;
        $membranes[$id]['fronts'] = sortPopulationByNonDomination($symbolObjs);
    }
}
